import { useState, useEffect, useRef, useCallback } from 'react';
import { gsap } from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';
import { ChevronLeft, ChevronRight, MapPin, Clock } from 'lucide-react';

gsap.registerPlugin(ScrollTrigger);

interface Slide {
  id: number;
  image: string;
  title: string;
  location: string;
  time: string;
  category: string;
}

const slides: Slide[] = [
  {
    id: 1,
    image: '/featured_airport.jpg',
    title: 'Airport Coffee at 5AM',
    location: 'Atlanta, GA',
    time: '5:07 AM',
    category: 'Travel',
  },
  {
    id: 2,
    image: '/featured_stone_mountain.jpg',
    title: 'Stone Mountain Silence',
    location: 'Atlanta, GA',
    time: '5:15 AM',
    category: 'Hiking',
  },
  {
    id: 3,
    image: '/featured_purple_sky.jpg',
    title: 'Purple Sky, No Reason',
    location: 'Denver, CO',
    time: '5:22 AM',
    category: 'Reflection',
  },
  {
    id: 4,
    image: '/featured_maui.jpg',
    title: 'Maui First Light',
    location: 'Maui, HI',
    time: '5:41 AM',
    category: 'Sunrise',
  },
  {
    id: 5,
    image: '/featured_city.jpg',
    title: 'City Still Asleep',
    location: 'New York, NY',
    time: '5:56 AM',
    category: 'City',
  },
  {
    id: 6,
    image: '/featured_trailhead.jpg',
    title: 'Trailhead Headlamp Hour',
    location: 'Sedona, AZ',
    time: '4:48 AM',
    category: 'Hiking',
  },
];

const FeaturedSlider = () => {
  const [currentSlide, setCurrentSlide] = useState(0);
  const [isPaused, setIsPaused] = useState(false);
  const sectionRef = useRef<HTMLElement>(null);
  const sliderRef = useRef<HTMLDivElement>(null);
  const trackRef = useRef<HTMLDivElement>(null);

  const nextSlide = useCallback(() => {
    setCurrentSlide((prev) => (prev + 1) % slides.length);
  }, []);

  const prevSlide = useCallback(() => {
    setCurrentSlide((prev) => (prev - 1 + slides.length) % slides.length);
  }, []);

  // Auto-play
  useEffect(() => {
    if (isPaused) return;
    const interval = setInterval(nextSlide, 5000);
    return () => clearInterval(interval);
  }, [isPaused, nextSlide]);

  // Keyboard navigation
  useEffect(() => {
    const handleKeyDown = (e: KeyboardEvent) => {
      if (e.key === 'ArrowLeft') prevSlide();
      if (e.key === 'ArrowRight') nextSlide();
    };
    window.addEventListener('keydown', handleKeyDown);
    return () => window.removeEventListener('keydown', handleKeyDown);
  }, [nextSlide, prevSlide]);

  // Scroll animation
  useEffect(() => {
    const ctx = gsap.context(() => {
      // Section entrance
      gsap.fromTo(
        sectionRef.current,
        { opacity: 0, y: 60 },
        {
          opacity: 1,
          y: 0,
          duration: 0.8,
          scrollTrigger: {
            trigger: sectionRef.current,
            start: 'top 80%',
            end: 'top 55%',
            scrub: 1,
          },
        }
      );

      // Slider stage entrance
      gsap.fromTo(
        sliderRef.current,
        { opacity: 0, y: 40, scale: 0.96 },
        {
          opacity: 1,
          y: 0,
          scale: 1,
          duration: 0.8,
          scrollTrigger: {
            trigger: sliderRef.current,
            start: 'top 80%',
            end: 'top 55%',
            scrub: 1,
          },
        }
      );
    }, sectionRef);

    return () => ctx.revert();
  }, []);

  // Slide transition
  useEffect(() => {
    if (trackRef.current) {
      gsap.to(trackRef.current, {
        x: `-${currentSlide * 100}%`,
        duration: 0.6,
        ease: 'power2.inOut',
      });
    }
  }, [currentSlide]);

  return (
    <section
      id="featured"
      ref={sectionRef}
      className="relative w-full py-20 lg:py-32 bg-[#F6F4EF] z-20"
    >
      <div className="w-full px-6 lg:px-12">
        {/* Header */}
        <div className="flex items-center justify-between mb-8 lg:mb-12">
          <span className="section-label">Featured</span>
          <div className="flex items-center gap-2">
            <button
              onClick={prevSlide}
              className="p-2 rounded-full border border-[#1A1D23]/10 text-[#1A1D23] hover:bg-[#1A1D23] hover:text-white transition-all duration-300"
              aria-label="Previous slide"
            >
              <ChevronLeft className="w-5 h-5" />
            </button>
            <button
              onClick={nextSlide}
              className="p-2 rounded-full border border-[#1A1D23]/10 text-[#1A1D23] hover:bg-[#1A1D23] hover:text-white transition-all duration-300"
              aria-label="Next slide"
            >
              <ChevronRight className="w-5 h-5" />
            </button>
          </div>
        </div>

        {/* Slider */}
        <div
          ref={sliderRef}
          className="relative overflow-hidden rounded-xl"
          onMouseEnter={() => setIsPaused(true)}
          onMouseLeave={() => setIsPaused(false)}
        >
          {/* Track */}
          <div
            ref={trackRef}
            className="flex transition-transform"
            style={{ width: `${slides.length * 100}%` }}
          >
            {slides.map((slide) => (
              <div
                key={slide.id}
                className="relative w-full flex-shrink-0"
                style={{ width: `${100 / slides.length}%` }}
              >
                <div className="relative aspect-[16/9] lg:aspect-[21/9] overflow-hidden rounded-xl">
                  <img
                    src={slide.image}
                    alt={slide.title}
                    className="w-full h-full object-cover"
                  />
                  {/* Scrim */}
                  <div className="absolute inset-0 image-scrim" />
                  
                  {/* Content */}
                  <div className="absolute inset-0 flex flex-col justify-end p-6 lg:p-10">
                    {/* Category badge */}
                    <span className="inline-block self-start px-3 py-1 mb-4 text-xs font-mono uppercase tracking-wider bg-white/20 backdrop-blur-sm text-white rounded-full">
                      {slide.category}
                    </span>
                    
                    {/* Title */}
                    <h3 className="text-white text-2xl sm:text-3xl lg:text-4xl font-bold mb-3">
                      {slide.title}
                    </h3>
                    
                    {/* Meta */}
                    <div className="flex items-center gap-4 text-white/70 text-sm">
                      <span className="flex items-center gap-1.5">
                        <MapPin className="w-4 h-4" />
                        {slide.location}
                      </span>
                      <span className="flex items-center gap-1.5">
                        <Clock className="w-4 h-4" />
                        {slide.time}
                      </span>
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>

          {/* Card shadow */}
          <div className="absolute inset-0 rounded-xl pointer-events-none card-shadow" />
        </div>

        {/* Dots */}
        <div className="flex items-center justify-center gap-2 mt-6">
          {slides.map((_, index) => (
            <button
              key={index}
              onClick={() => setCurrentSlide(index)}
              className={`w-2 h-2 rounded-full transition-all duration-300 ${
                index === currentSlide
                  ? 'w-6 bg-[#D4A27F]'
                  : 'bg-[#1A1D23]/20 hover:bg-[#1A1D23]/40'
              }`}
              aria-label={`Go to slide ${index + 1}`}
            />
          ))}
        </div>
      </div>
    </section>
  );
};

export default FeaturedSlider;
