import { useRef, useEffect, useState } from 'react';
import { gsap } from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';
import { Sun, Instagram, ArrowRight } from 'lucide-react';

gsap.registerPlugin(ScrollTrigger);

const Footer = () => {
  const sectionRef = useRef<HTMLElement>(null);
  const contentRef = useRef<HTMLDivElement>(null);
  const bgRef = useRef<HTMLDivElement>(null);
  const [email, setEmail] = useState('');

  useEffect(() => {
    const ctx = gsap.context(() => {
      const scrollTl = gsap.timeline({
        scrollTrigger: {
          trigger: sectionRef.current,
          start: 'top top',
          end: '+=130%',
          pin: true,
          scrub: 0.6,
        },
      });

      // Background animation
      if (bgRef.current) {
        scrollTl.fromTo(
          bgRef.current,
          { scale: 1.08, opacity: 0.8 },
          { scale: 1, opacity: 1, ease: 'none' },
          0
        );
      }

      // Content entrance
      if (contentRef.current) {
        scrollTl.fromTo(
          contentRef.current,
          { opacity: 0, y: 40 },
          { opacity: 1, y: 0, ease: 'none' },
          0.1
        );

        // Exit animation
        scrollTl.to(
          contentRef.current,
          { y: '-10vh', opacity: 0, ease: 'power2.in' },
          0.7
        );
      }
    }, sectionRef);

    return () => ctx.revert();
  }, []);

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (email) {
      alert('Thank you for subscribing to 5AM moments!');
      setEmail('');
    }
  };

  const footerLinks = [
    { label: 'About', href: '#' },
    { label: 'Contact', href: '#' },
    { label: 'Privacy', href: '#' },
    { label: 'Terms', href: '#' },
  ];

  return (
    <section
      ref={sectionRef}
      className="relative w-full h-screen overflow-hidden z-40"
    >
      {/* Background Image */}
      <div ref={bgRef} className="absolute inset-0 w-full h-full">
        <img
          src="/closing_night_sky.jpg"
          alt="Night sky"
          className="w-full h-full object-cover"
        />
        {/* Overlay */}
        <div className="absolute inset-0 bg-[#10141B]/60" />
      </div>

      {/* Content */}
      <div
        ref={contentRef}
        className="relative z-10 flex flex-col items-center justify-center h-full px-6"
      >
        <div className="text-center max-w-xl mx-auto">
          {/* Logo */}
          <div className="flex items-center justify-center gap-2 mb-6">
            <Sun className="w-6 h-6 text-[#D4A27F]" />
            <span className="text-white font-bold text-xl tracking-tight">5AM</span>
          </div>

          <h2 className="text-white text-3xl sm:text-4xl lg:text-5xl font-black uppercase tracking-tight mb-4">
            See You Tomorrow.
          </h2>

          <p className="text-white/70 text-base lg:text-lg mb-8">
            Get one sunrise moment in your inbox—no noise, just the quiet.
          </p>

          {/* Newsletter Form */}
          <form onSubmit={handleSubmit} className="flex flex-col sm:flex-row gap-3 mb-12">
            <input
              type="email"
              value={email}
              onChange={(e) => setEmail(e.target.value)}
              placeholder="Email address"
              className="flex-1 px-5 py-3.5 rounded-full bg-white/10 backdrop-blur-sm border border-white/20 text-white placeholder:text-white/50 focus:outline-none focus:border-[#D4A27F] focus:ring-1 focus:ring-[#D4A27F] transition-all duration-300"
            />
            <button
              type="submit"
              className="px-6 py-3.5 bg-[#D4A27F] text-[#1A1D23] rounded-full font-medium hover:bg-[#C49470] transition-colors duration-300 flex items-center justify-center gap-2"
            >
              Subscribe
              <ArrowRight className="w-4 h-4" />
            </button>
          </form>

          {/* Footer Links */}
          <div className="flex flex-wrap items-center justify-center gap-6 mb-8">
            {footerLinks.map((link) => (
              <a
                key={link.label}
                href={link.href}
                className="text-white/60 text-sm hover:text-white transition-colors duration-300"
              >
                {link.label}
              </a>
            ))}
            <a
              href="https://instagram.com"
              target="_blank"
              rel="noopener noreferrer"
              className="text-white/60 hover:text-white transition-colors duration-300"
            >
              <Instagram className="w-5 h-5" />
            </a>
          </div>

          {/* Copyright */}
          <p className="text-white/40 text-xs">
            © {new Date().getFullYear()} 5AM In The Morning. All rights reserved.
          </p>
        </div>
      </div>
    </section>
  );
};

export default Footer;
