import { useEffect, useRef } from 'react';
import { gsap } from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';
import { ChevronDown } from 'lucide-react';

gsap.registerPlugin(ScrollTrigger);

const Hero = () => {
  const sectionRef = useRef<HTMLElement>(null);
  const contentRef = useRef<HTMLDivElement>(null);
  const headlineRef = useRef<HTMLHeadingElement>(null);
  const sublineRef = useRef<HTMLParagraphElement>(null);
  const ctaRef = useRef<HTMLDivElement>(null);
  const bgRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const ctx = gsap.context(() => {
      // Initial load animation
      const loadTl = gsap.timeline({ defaults: { ease: 'power3.out' } });

      // Background fade in with scale
      loadTl.fromTo(
        bgRef.current,
        { opacity: 0, scale: 1.06 },
        { opacity: 1, scale: 1, duration: 1.2 }
      );

      // Headline word-by-word reveal
      if (headlineRef.current) {
        const words = headlineRef.current.querySelectorAll('.word');
        loadTl.fromTo(
          words,
          { opacity: 0, y: 24, rotateX: 35 },
          { opacity: 1, y: 0, rotateX: 0, duration: 0.6, stagger: 0.05 },
          '-=0.8'
        );
      }

      // Subline fade up
      loadTl.fromTo(
        sublineRef.current,
        { opacity: 0, y: 14 },
        { opacity: 1, y: 0, duration: 0.6 },
        '-=0.4'
      );

      // CTAs fade up
      loadTl.fromTo(
        ctaRef.current?.children || [],
        { opacity: 0, y: 18 },
        { opacity: 1, y: 0, duration: 0.5, stagger: 0.08 },
        '-=0.3'
      );

      // Scroll-driven exit animation
      const scrollTl = gsap.timeline({
        scrollTrigger: {
          trigger: sectionRef.current,
          start: 'top top',
          end: '+=130%',
          pin: true,
          scrub: 0.6,
          onLeaveBack: () => {
            // Reset all elements when scrolling back to top
            gsap.set([headlineRef.current, sublineRef.current, ctaRef.current], {
              opacity: 1,
              y: 0,
            });
            gsap.set(bgRef.current, { scale: 1, y: 0 });
          },
        },
      });

      // ENTRANCE (0%-30%): Hold - already visible from load animation
      // SETTLE (30%-70%): Hold
      // EXIT (70%-100%): Fade out and move up

      // Content exit
      scrollTl.fromTo(
        contentRef.current,
        { y: 0, opacity: 1 },
        { y: '-22vh', opacity: 0, ease: 'power2.in' },
        0.7
      );

      // Background parallax
      scrollTl.fromTo(
        bgRef.current,
        { scale: 1, y: 0 },
        { scale: 1.08, y: '-6vh' },
        0.7
      );
    }, sectionRef);

    return () => ctx.revert();
  }, []);

  return (
    <section
      ref={sectionRef}
      className="relative w-full h-screen overflow-hidden z-10"
    >
      {/* Background Image */}
      <div
        ref={bgRef}
        className="absolute inset-0 w-full h-full"
        style={{ opacity: 0 }}
      >
        <img
          src="/hero_night_sky.jpg"
          alt="Night sky with stars"
          className="w-full h-full object-cover"
        />
        {/* Gradient overlay */}
        <div className="absolute inset-0 bg-gradient-to-b from-[#10141B]/35 via-[#10141B]/45 to-[#10141B]/60" />
      </div>

      {/* Content */}
      <div
        ref={contentRef}
        className="relative z-10 flex flex-col items-center justify-center h-full px-6 text-center"
      >
        <h1
          ref={headlineRef}
          className="text-white text-4xl sm:text-5xl md:text-6xl lg:text-7xl xl:text-8xl font-black uppercase tracking-tight mb-6"
          style={{ perspective: '1000px' }}
        >
          <span className="word inline-block">Why</span>{' '}
          <span className="word inline-block">are</span>{' '}
          <span className="word inline-block">you</span>{' '}
          <span className="word inline-block">awake?</span>
        </h1>

        <p
          ref={sublineRef}
          className="text-white/80 text-base sm:text-lg md:text-xl max-w-2xl mb-10 font-light"
          style={{ opacity: 0 }}
        >
          Sunrises. Stories. Quiet moments from people who woke up before the world did.
        </p>

        <div ref={ctaRef} className="flex flex-col sm:flex-row gap-4">
          <a href="#featured" className="btn-primary">
            Explore Today&apos;s Sunrises
          </a>
          <a href="#submit" className="btn-secondary border-white/30 text-white hover:bg-white hover:text-[#1A1D23]">
            Submit Yours
          </a>
        </div>
      </div>

      {/* Bottom elements */}
      <div className="absolute bottom-8 left-0 right-0 z-10 flex items-end justify-between px-6 lg:px-12">
        <p className="text-white/50 text-xs font-mono uppercase tracking-wider">
          Scroll to enter the quiet.
        </p>
        <ChevronDown className="w-5 h-5 text-white/50 animate-bounce" />
      </div>
    </section>
  );
};

export default Hero;
