import { useRef, useEffect, useState } from 'react';
import { gsap } from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';
import { Send, Image as ImageIcon, X } from 'lucide-react';

gsap.registerPlugin(ScrollTrigger);

const SubmitForm = () => {
  const sectionRef = useRef<HTMLElement>(null);
  const formRef = useRef<HTMLDivElement>(null);
  const [dragActive, setDragActive] = useState(false);
  const [previewImage, setPreviewImage] = useState<string | null>(null);
  const [formData, setFormData] = useState({
    title: '',
    location: '',
    time: '',
    story: '',
  });

  useEffect(() => {
    const ctx = gsap.context(() => {
      // Form card entrance
      gsap.fromTo(
        formRef.current,
        { opacity: 0, y: 60, scale: 0.98 },
        {
          opacity: 1,
          y: 0,
          scale: 1,
          duration: 0.8,
          scrollTrigger: {
            trigger: sectionRef.current,
            start: 'top 80%',
            end: 'top 55%',
            scrub: 1,
          },
        }
      );

      // Fields stagger animation
      const fields = formRef.current?.querySelectorAll('.form-field');
      if (fields) {
        gsap.fromTo(
          fields,
          { opacity: 0, y: 20 },
          {
            opacity: 1,
            y: 0,
            duration: 0.4,
            stagger: 0.05,
            scrollTrigger: {
              trigger: formRef.current,
              start: 'top 70%',
              toggleActions: 'play none none reverse',
            },
          }
        );
      }
    }, sectionRef);

    return () => ctx.revert();
  }, []);

  const handleDrag = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (e.type === 'dragenter' || e.type === 'dragover') {
      setDragActive(true);
    } else if (e.type === 'dragleave') {
      setDragActive(false);
    }
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setDragActive(false);
    
    if (e.dataTransfer.files && e.dataTransfer.files[0]) {
      const file = e.dataTransfer.files[0];
      if (file.type.startsWith('image/')) {
        const reader = new FileReader();
        reader.onload = (e) => {
          setPreviewImage(e.target?.result as string);
        };
        reader.readAsDataURL(file);
      }
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      const reader = new FileReader();
      reader.onload = (e) => {
        setPreviewImage(e.target?.result as string);
      };
      reader.readAsDataURL(file);
    }
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    // Handle form submission
    alert('Thank you for sharing your 5AM moment!');
  };

  return (
    <section
      id="submit"
      ref={sectionRef}
      className="relative w-full py-20 lg:py-32 bg-[#F6F4EF] z-20"
    >
      <div className="w-full px-6 lg:px-12">
        <div
          ref={formRef}
          className="max-w-3xl mx-auto bg-white rounded-xl border border-[#1A1D23]/5 p-6 sm:p-8 lg:p-12"
        >
          {/* Header */}
          <div className="text-center mb-8 lg:mb-10">
            <h2 className="text-3xl sm:text-4xl lg:text-5xl font-black text-[#1A1D23] uppercase tracking-tight mb-4">
              Submit Your 5AM
            </h2>
            <p className="text-[#6E727A] text-base lg:text-lg">
              Tell us where you were, what you saw, and why you were awake.
            </p>
          </div>

          {/* Form */}
          <form onSubmit={handleSubmit} className="space-y-6">
            {/* Title */}
            <div className="form-field">
              <label className="block text-sm font-medium text-[#1A1D23] mb-2">
                Title
              </label>
              <input
                type="text"
                value={formData.title}
                onChange={(e) => setFormData({ ...formData, title: e.target.value })}
                placeholder="e.g., Airport Coffee at 5AM"
                className="w-full px-4 py-3 rounded-lg border border-[#1A1D23]/10 bg-transparent text-[#1A1D23] placeholder:text-[#6E727A]/50 focus:outline-none focus:border-[#D4A27F] focus:ring-1 focus:ring-[#D4A27F] transition-all duration-300"
              />
            </div>

            {/* Location & Time */}
            <div className="form-field grid grid-cols-1 sm:grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-[#1A1D23] mb-2">
                  Location
                </label>
                <input
                  type="text"
                  value={formData.location}
                  onChange={(e) => setFormData({ ...formData, location: e.target.value })}
                  placeholder="e.g., Atlanta, GA"
                  className="w-full px-4 py-3 rounded-lg border border-[#1A1D23]/10 bg-transparent text-[#1A1D23] placeholder:text-[#6E727A]/50 focus:outline-none focus:border-[#D4A27F] focus:ring-1 focus:ring-[#D4A27F] transition-all duration-300"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-[#1A1D23] mb-2">
                  Time
                </label>
                <input
                  type="text"
                  value={formData.time}
                  onChange={(e) => setFormData({ ...formData, time: e.target.value })}
                  placeholder="e.g., 5:07 AM"
                  className="w-full px-4 py-3 rounded-lg border border-[#1A1D23]/10 bg-transparent text-[#1A1D23] placeholder:text-[#6E727A]/50 focus:outline-none focus:border-[#D4A27F] focus:ring-1 focus:ring-[#D4A27F] transition-all duration-300"
                />
              </div>
            </div>

            {/* Story */}
            <div className="form-field">
              <label className="block text-sm font-medium text-[#1A1D23] mb-2">
                Your Story
              </label>
              <textarea
                value={formData.story}
                onChange={(e) => setFormData({ ...formData, story: e.target.value })}
                placeholder="What did you see? Why were you awake?"
                rows={4}
                className="w-full px-4 py-3 rounded-lg border border-[#1A1D23]/10 bg-transparent text-[#1A1D23] placeholder:text-[#6E727A]/50 focus:outline-none focus:border-[#D4A27F] focus:ring-1 focus:ring-[#D4A27F] transition-all duration-300 resize-none"
              />
            </div>

            {/* Image Upload */}
            <div className="form-field">
              <label className="block text-sm font-medium text-[#1A1D23] mb-2">
                Add a Photo
              </label>
              
              {previewImage ? (
                <div className="relative rounded-lg overflow-hidden">
                  <img
                    src={previewImage}
                    alt="Preview"
                    className="w-full h-48 object-cover"
                  />
                  <button
                    type="button"
                    onClick={() => setPreviewImage(null)}
                    className="absolute top-2 right-2 p-1.5 bg-[#1A1D23]/70 text-white rounded-full hover:bg-[#1A1D23] transition-colors duration-300"
                  >
                    <X className="w-4 h-4" />
                  </button>
                </div>
              ) : (
                <div
                  onDragEnter={handleDrag}
                  onDragLeave={handleDrag}
                  onDragOver={handleDrag}
                  onDrop={handleDrop}
                  className={`relative border-2 border-dashed rounded-lg p-8 text-center transition-all duration-300 ${
                    dragActive
                      ? 'border-[#D4A27F] bg-[#D4A27F]/5'
                      : 'border-[#1A1D23]/10 hover:border-[#D4A27F]/50'
                  }`}
                >
                  <input
                    type="file"
                    accept="image/*"
                    onChange={handleFileChange}
                    className="absolute inset-0 w-full h-full opacity-0 cursor-pointer"
                  />
                  <div className="flex flex-col items-center gap-3">
                    <div className="w-12 h-12 rounded-full bg-[#F6F4EF] flex items-center justify-center">
                      <ImageIcon className="w-5 h-5 text-[#D4A27F]" />
                    </div>
                    <div>
                      <p className="text-sm text-[#1A1D23] font-medium">
                        Drop your photo here, or click to browse
                      </p>
                      <p className="text-xs text-[#6E727A] mt-1">
                        Supports JPG, PNG up to 10MB
                      </p>
                    </div>
                  </div>
                </div>
              )}
            </div>

            {/* Submit Button */}
            <div className="form-field pt-4">
              <button
                type="submit"
                className="w-full btn-primary flex items-center justify-center gap-2 py-4"
              >
                <Send className="w-4 h-4" />
                Send Moment
              </button>
            </div>
          </form>
        </div>
      </div>
    </section>
  );
};

export default SubmitForm;
